<?php
/**
 * API لجلب بيانات التقارير
 * يستخدم هذا الملف لجلب بيانات التقارير الأسبوعية والشهرية
 * مع مراعاة صلاحيات المستخدم
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// بدء الجلسة إذا لم تكن قد بدأت بالفعل
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// تضمين ملف الاتصال بقاعدة البيانات
require_once '../config/db_connection.php';

// التحقق من أن الطلب هو طلب GET
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'يجب استخدام طريقة GET']);
    exit;
}

// التحقق من تسجيل دخول المستخدم
if (!isset($_SESSION['user_id'])) {
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'يجب تسجيل الدخول للوصول إلى هذه البيانات']);
    exit;
}

// الحصول على معلومات المستخدم
$userId = $_SESSION['user_id'];
$userRole = $_SESSION['role'] ?? 'user';

// الحصول على نوع التقرير (أسبوعي أو شهري)
$reportType = isset($_GET['report_type']) ? $_GET['report_type'] : 'weekly';

// الحصول على نوع البيانات (مستخدم أو إدارة)
$dataType = isset($_GET['data_type']) ? $_GET['data_type'] : 'user';

// التحقق من الصلاحيات للتقارير الإدارية
if ($dataType === 'admin' && $userRole !== 'admin' && $userRole !== 'manager') {
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'ليس لديك صلاحية للوصول إلى تقارير الإدارة']);
    exit;
}

// الحصول على معلمات التصفية
$year = isset($_GET['year']) ? (int)$_GET['year'] : (int)date('Y');
$period = isset($_GET['period']) ? (int)$_GET['period'] : ($reportType === 'weekly' ? (int)date('W') : (int)date('n'));
$branch = isset($_GET['branch']) ? $_GET['branch'] : 'all';

try {
    // تحديد فترة التقرير
    if ($reportType === 'weekly') {
        // تحديد تاريخ بداية ونهاية الأسبوع
        $dto = new DateTime();
        $dto->setISODate($year, $period);
        $startDate = $dto->format('Y-m-d');
        $dto->modify('+6 days');
        $endDate = $dto->format('Y-m-d');
    } else {
        // تحديد تاريخ بداية ونهاية الشهر
        $startDate = date('Y-m-d', strtotime("$year-$period-01"));
        $endDate = date('Y-m-t', strtotime($startDate));
    }

    // بناء استعلام SQL
    $sql = "
        SELECT
            COUNT(*) as total_orders,
            SUM(CASE WHEN status = 'تم التسليم' THEN 1 ELSE 0 END) as completed_orders,
            SUM(CASE WHEN status = 'قيد المعالجة' THEN 1 ELSE 0 END) as pending_orders,
            SUM(CASE WHEN status = 'في الطريق' THEN 1 ELSE 0 END) as in_transit_orders,
            SUM(CASE WHEN status = 'ملغى' THEN 1 ELSE 0 END) as cancelled_orders,
            SUM(amount_agreed_with_customer) as total_revenue,
            SUM(profit) as total_profit,
            DATE(order_upload_date) as order_date
        FROM orders FORCE INDEX (idx_orders_upload_date, idx_orders_status)
        WHERE order_upload_date BETWEEN ? AND ?
    ";

    // إضافة شروط التصفية
    $params = [$startDate, $endDate];

    // تقييد البيانات للمستخدمين العاديين (غير المديرين)
    if ($dataType === 'user' && $userRole !== 'admin' && $userRole !== 'manager') {
        $sql .= " AND user_id = ?";
        $params[] = $userId;
    }

    // تصفية حسب الفرع
    if ($branch !== 'all') {
        $sql .= " AND branch_id = (SELECT id FROM branches WHERE name = ?)";
        $params[] = $branch;
    }

    // تجميع البيانات حسب التاريخ
    $sql .= " GROUP BY order_date ORDER BY order_date";

    // تنفيذ الاستعلام
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    $dailyData = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // استعلام لإجمالي الإحصائيات
    $totalSql = "
        SELECT
            COUNT(*) as total_orders,
            SUM(CASE WHEN status = 'تم التسليم' THEN 1 ELSE 0 END) as completed_orders,
            SUM(CASE WHEN status = 'قيد المعالجة' THEN 1 ELSE 0 END) as pending_orders,
            SUM(CASE WHEN status = 'في الطريق' THEN 1 ELSE 0 END) as in_transit_orders,
            SUM(CASE WHEN status = 'ملغى' THEN 1 ELSE 0 END) as cancelled_orders,
            SUM(amount_agreed_with_customer) as total_revenue,
            SUM(profit) as total_profit,
            COUNT(DISTINCT DATE(order_upload_date)) as total_days,
            COUNT(*) / GREATEST(DATEDIFF(?, ?), 1) as daily_average
        FROM orders FORCE INDEX (idx_orders_upload_date, idx_orders_status)
        WHERE order_upload_date BETWEEN ? AND ?
    ";

    // إضافة نفس شروط التصفية
    if ($dataType === 'user' && $userRole !== 'admin' && $userRole !== 'manager') {
        $totalSql .= " AND user_id = ?";
    }

    if ($branch !== 'all') {
        $totalSql .= " AND branch_id = (SELECT id FROM branches WHERE name = ?)";
    }

    // إضافة معلمات التاريخ للمتوسط اليومي
    $totalParams = array_merge([$endDate, $startDate], $params);

    // تنفيذ استعلام الإجمالي
    $totalStmt = $conn->prepare($totalSql);
    $totalStmt->execute($totalParams);
    $totalStats = $totalStmt->fetch(PDO::FETCH_ASSOC);

    // استعلام لجلب قائمة الطلبات الأسبوعية
    $ordersSql = "
        SELECT
            o.id,
            o.order_number,
            o.client_name,
            o.service_type,
            b.name as branch_name,
            o.departure_area,
            o.second_destination,
            o.delivery_area,
            o.goods_name,
            o.goods_type,
            o.order_upload_date,
            o.scheduled_delivery_date,
            o.amount_agreed_with_customer,
            o.payment_status,
            o.status,
            o.employee_name
        FROM orders o FORCE INDEX (idx_orders_upload_date)
        LEFT JOIN branches b ON o.branch_id = b.id
        WHERE o.order_upload_date BETWEEN ? AND ?
    ";

    // إضافة نفس شروط التصفية
    $ordersParams = $params;

    if ($dataType === 'user' && $userRole !== 'admin' && $userRole !== 'manager') {
        $ordersSql .= " AND o.user_id = ?";
    }

    if ($branch !== 'all') {
        $ordersSql .= " AND o.branch_id = (SELECT id FROM branches WHERE name = ?)";
    }

    $ordersSql .= " ORDER BY o.order_upload_date DESC";

    // تنفيذ استعلام الطلبات
    $ordersStmt = $conn->prepare($ordersSql);
    $ordersStmt->execute($ordersParams);
    $weeklyOrders = $ordersStmt->fetchAll(PDO::FETCH_ASSOC);

    // إذا كان التقرير شهريًا، نقوم بإعداد بيانات الأسابيع
    $weeklyData = [];

    if ($reportType === 'monthly') {
        // استعلام لجلب بيانات الأسابيع في الشهر
        $weeklyDataSql = "
            SELECT
                WEEK(order_upload_date) as week_number,
                COUNT(*) as orders_count,
                SUM(amount_agreed_with_customer) as total_revenue
            FROM orders FORCE INDEX (idx_orders_upload_date)
            WHERE order_upload_date BETWEEN ? AND ?
        ";

        // إضافة نفس شروط التصفية
        $weeklyDataParams = $params;

        if ($dataType === 'user' && $userRole !== 'admin' && $userRole !== 'manager') {
            $weeklyDataSql .= " AND user_id = ?";
        }

        if ($branch !== 'all') {
            $weeklyDataSql .= " AND branch_id = (SELECT id FROM branches WHERE name = ?)";
        }

        $weeklyDataSql .= " GROUP BY WEEK(order_upload_date) ORDER BY week_number";

        // تنفيذ استعلام بيانات الأسابيع
        $weeklyDataStmt = $conn->prepare($weeklyDataSql);
        $weeklyDataStmt->execute($weeklyDataParams);
        $weeklyDataResults = $weeklyDataStmt->fetchAll(PDO::FETCH_ASSOC);

        // تحويل البيانات إلى التنسيق المطلوب
        foreach ($weeklyDataResults as $index => $weekData) {
            $weeklyData[] = [
                'week' => 'الأسبوع ' . ($index + 1),
                'week_number' => $weekData['week_number'],
                'orders' => (int)$weekData['orders_count'],
                'revenue' => (float)$weekData['total_revenue']
            ];
        }

        // إضافة أسابيع فارغة إذا كان عدد الأسابيع أقل من 4
        while (count($weeklyData) < 4) {
            $weeklyData[] = [
                'week' => 'الأسبوع ' . (count($weeklyData) + 1),
                'week_number' => 0,
                'orders' => 0,
                'revenue' => 0
            ];
        }

        // استعلام لجلب بيانات الفروع
        $branchesDataSql = "
            SELECT
                b.name as branch_name,
                COUNT(*) as orders_count,
                COUNT(*) * 100.0 / (SELECT COUNT(*) FROM orders WHERE order_upload_date BETWEEN ? AND ?) as percentage
            FROM orders o FORCE INDEX (idx_orders_upload_date, idx_orders_branch_id)
            LEFT JOIN branches b ON o.branch_id = b.id
            WHERE o.order_upload_date BETWEEN ? AND ?
        ";

        // إضافة نفس شروط التصفية
        $branchesDataParams = array_merge($params, $params);

        if ($dataType === 'user' && $userRole !== 'admin' && $userRole !== 'manager') {
            $branchesDataSql .= " AND o.user_id = ?";
            $branchesDataParams[] = $userId;
        }

        $branchesDataSql .= " GROUP BY b.name ORDER BY orders_count DESC LIMIT 5";

        // تنفيذ استعلام بيانات الفروع
        $branchesDataStmt = $conn->prepare($branchesDataSql);
        $branchesDataStmt->execute($branchesDataParams);
        $branchesData = $branchesDataStmt->fetchAll(PDO::FETCH_ASSOC);

        // تحويل البيانات إلى التنسيق المطلوب
        $formattedBranchesData = [];
        foreach ($branchesData as $branch) {
            $formattedBranchesData[] = [
                'name' => $branch['branch_name'] ?: 'غير محدد',
                'orders' => (int)$branch['orders_count'],
                'percentage' => round((float)$branch['percentage'])
            ];
        }

        // إضافة فروع فارغة إذا كان عدد الفروع أقل من 5
        while (count($formattedBranchesData) < 5) {
            $formattedBranchesData[] = [
                'name' => 'فرع ' . (count($formattedBranchesData) + 1),
                'orders' => 0,
                'percentage' => 0
            ];
        }

        // استعلام لجلب بيانات الشهر السابق
        $prevMonthSql = "
            SELECT
                COUNT(*) as total_orders
            FROM orders FORCE INDEX (idx_orders_upload_date)
            WHERE order_upload_date BETWEEN ? AND ?
        ";

        // تحديد تاريخ بداية ونهاية الشهر السابق
        $prevMonthStart = date('Y-m-d', strtotime('first day of previous month', strtotime($startDate)));
        $prevMonthEnd = date('Y-m-d', strtotime('last day of previous month', strtotime($startDate)));

        // تنفيذ استعلام الشهر السابق
        $prevMonthStmt = $conn->prepare($prevMonthSql);
        $prevMonthStmt->execute([$prevMonthStart, $prevMonthEnd]);
        $prevMonthData = $prevMonthStmt->fetch(PDO::FETCH_ASSOC);

        // حساب نسبة النمو
        $prevMonthOrders = (int)$prevMonthData['total_orders'];
        $totalGrowth = $prevMonthOrders > 0 ? round((($totalStats['total_orders'] - $prevMonthOrders) / $prevMonthOrders) * 100) : 100;
    }

    // تنسيق البيانات للعرض
    $formattedData = [
        'period' => [
            'start' => $startDate,
            'end' => $endDate,
            'type' => $reportType,
            'value' => $period,
            'year' => $year
        ],
        'stats' => [
            'total_orders' => (int)$totalStats['total_orders'],
            'completed_orders' => (int)$totalStats['completed_orders'],
            'pending_orders' => (int)$totalStats['pending_orders'],
            'in_transit_orders' => (int)$totalStats['in_transit_orders'],
            'cancelled_orders' => (int)$totalStats['cancelled_orders'],
            'completed_percentage' => $totalStats['total_orders'] > 0 ? round(($totalStats['completed_orders'] / $totalStats['total_orders']) * 100) : 0,
            'pending_percentage' => $totalStats['total_orders'] > 0 ? round(($totalStats['pending_orders'] / $totalStats['total_orders']) * 100) : 0,
            'in_transit_percentage' => $totalStats['total_orders'] > 0 ? round(($totalStats['in_transit_orders'] / $totalStats['total_orders']) * 100) : 0,
            'cancelled_percentage' => $totalStats['total_orders'] > 0 ? round(($totalStats['cancelled_orders'] / $totalStats['total_orders']) * 100) : 0,
            'total_revenue' => (float)$totalStats['total_revenue'],
            'total_profit' => (float)$totalStats['total_profit'],
            'daily_average' => round((float)$totalStats['daily_average'], 1),
            'total_days' => (int)$totalStats['total_days']
        ],
        'daily_data' => $dailyData,
        'orders' => $weeklyOrders
    ];

    // إضافة بيانات خاصة بالتقارير الشهرية
    if ($reportType === 'monthly') {
        $formattedData['stats']['total_growth'] = $totalGrowth;
        $formattedData['stats']['prev_month_orders'] = $prevMonthOrders;
        $formattedData['weekly_data'] = $weeklyData;
        $formattedData['branches'] = $formattedBranchesData;
    }

    // إرجاع البيانات
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'success',
        'data' => $formattedData
    ]);

} catch (PDOException $e) {
    // إرجاع رسالة الخطأ
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'فشل جلب بيانات التقرير: ' . $e->getMessage()]);
}
?>
