<?php
/**
 * API لجلب الطلبات المؤرشفة
 * يستخدم هذا الملف لجلب الطلبات المؤرشفة من قاعدة البيانات
 * مع مراعاة صلاحيات المستخدم (متاح فقط للإدارة)
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// بدء الجلسة إذا لم تكن قد بدأت بالفعل
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// تضمين ملف الاتصال بقاعدة البيانات
require_once '../config/db_connection.php';

// التحقق من تسجيل دخول المستخدم
if (!isset($_SESSION['user_id'])) {
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'يجب تسجيل الدخول للوصول إلى هذه البيانات']);
    exit;
}

// التحقق من صلاحيات المستخدم (متاح فقط للإدارة)
$userRole = isset($_SESSION['role']) ? $_SESSION['role'] : 'user';
if ($userRole !== 'admin' && $userRole !== 'manager') {
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'ليس لديك صلاحية للوصول إلى الأرشيف']);
    exit;
}

// التحقق من أن الطلب هو طلب GET
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'يجب استخدام طريقة GET']);
    exit;
}

try {
    // الحصول على معلمات التصفية
    $year = isset($_GET['year']) ? $_GET['year'] : null;
    $month = isset($_GET['month']) ? $_GET['month'] : null;
    $branch = isset($_GET['branch']) ? $_GET['branch'] : null;
    $search = isset($_GET['search']) ? $_GET['search'] : null;
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 10;
    $offset = ($page - 1) * $limit;

    // بناء استعلام SQL
    $sql = "
        SELECT
            o.id, o.order_number, o.client_name, b.name as branch,
            o.departure_area, o.second_destination, o.delivery_area,
            o.goods_name, o.goods_type, o.country_of_origin,
            o.weight, o.quantity, o.unit, o.service_fees,
            o.currency, o.payment_type, o.delivery_duration,
            o.amount_paid, o.remaining_amount, o.order_upload_date,
            o.scheduled_delivery_date, o.recipient_name, o.phone_number,
            o.delivery_company, o.employee_name, o.notes,
            o.amount_agreed_with_customer, o.amount_agreed_with_agent,
            o.other_expenses, o.profit, o.payment_status, o.status,
            o.created_at, o.updated_at, o.archive_date,
            u.full_name as created_by_name
        FROM orders o
        LEFT JOIN branches b ON o.branch_id = b.id
        LEFT JOIN users u ON o.user_id = u.id
        WHERE o.archived = 1
    ";

    // إضافة شروط التصفية
    $params = [];

    if ($year) {
        $sql .= " AND YEAR(o.archive_date) = ?";
        $params[] = $year;
    }

    if ($month) {
        $sql .= " AND MONTH(o.archive_date) = ?";
        $params[] = $month;
    }

    if ($branch) {
        $sql .= " AND b.name = ?";
        $params[] = $branch;
    }

    if ($search) {
        $sql .= " AND (o.order_number LIKE ? OR o.client_name LIKE ? OR o.goods_name LIKE ?)";
        $searchParam = "%$search%";
        $params[] = $searchParam;
        $params[] = $searchParam;
        $params[] = $searchParam;
    }

    // استعلام لحساب إجمالي عدد الطلبات
    $countSql = "SELECT COUNT(*) as total FROM orders o LEFT JOIN branches b ON o.branch_id = b.id WHERE o.archived = 1";
    $countParams = [];

    if ($year) {
        $countSql .= " AND YEAR(o.archive_date) = ?";
        $countParams[] = $year;
    }

    if ($month) {
        $countSql .= " AND MONTH(o.archive_date) = ?";
        $countParams[] = $month;
    }

    if ($branch) {
        $countSql .= " AND b.name = ?";
        $countParams[] = $branch;
    }

    if ($search) {
        $countSql .= " AND (o.order_number LIKE ? OR o.client_name LIKE ? OR o.goods_name LIKE ?)";
        $searchParam = "%$search%";
        $countParams[] = $searchParam;
        $countParams[] = $searchParam;
        $countParams[] = $searchParam;
    }

    // تنفيذ استعلام العدد الإجمالي
    $countStmt = $conn->prepare($countSql);
    $countStmt->execute($countParams);
    $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];

    // إضافة ترتيب وحدود للاستعلام الرئيسي
    $sql .= " ORDER BY o.archive_date DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;

    // تنفيذ الاستعلام الرئيسي
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // تنسيق البيانات للعرض
    $formattedOrders = [];
    foreach ($orders as $order) {
        $formattedOrders[] = [
            'id' => $order['id'],
            'order_number' => $order['order_number'],
            'client_name' => $order['client_name'],
            'branch' => $order['branch'],
            'service_type' => $order['service_type'],
            'goods_name' => $order['goods_name'],
            'goods_type' => $order['goods_type'],
            'departure_area' => $order['departure_area'],
            'delivery_area' => $order['delivery_area'],
            'order_date' => $order['order_upload_date'],
            'delivery_date' => $order['scheduled_delivery_date'],
            'archive_date' => $order['archive_date'],
            'amount' => $order['amount_agreed_with_customer'],
            'currency' => $order['currency'],
            'payment_status' => $order['payment_status'],
            'status' => $order['status'],
            'created_by' => $order['created_by_name']
        ];
    }

    // إرجاع البيانات
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'success',
        'total' => $total,
        'page' => $page,
        'limit' => $limit,
        'totalPages' => ceil($total / $limit),
        'orders' => $formattedOrders
    ]);

} catch (PDOException $e) {
    // إرجاع رسالة الخطأ
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'فشل جلب الطلبات المؤرشفة: ' . $e->getMessage()]);
}
?>
