<?php
/**
 * ملف إضافة الفهارس لتحسين أداء قاعدة البيانات
 * 
 * يقوم هذا الملف بإضافة الفهارس اللازمة لتحسين أداء قاعدة البيانات
 * خاصة للاستعلامات المستخدمة في صفحة التقارير الأسبوعية
 * 
 * @version 1.0
 * @author فريق التطوير
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تضمين ملف الاتصال بقاعدة البيانات
require_once 'config/db_connection.php';

// دالة لإضافة فهرس إذا لم يكن موجودًا
function addIndexIfNotExists($conn, $tableName, $indexName, $columns) {
    try {
        // التحقق من وجود الفهرس
        $stmt = $conn->prepare("
            SELECT COUNT(*) as index_exists
            FROM information_schema.statistics
            WHERE table_schema = DATABASE()
            AND table_name = ?
            AND index_name = ?
        ");
        $stmt->execute([$tableName, $indexName]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // إذا لم يكن الفهرس موجودًا، قم بإضافته
        if ($result['index_exists'] == 0) {
            $sql = "CREATE INDEX $indexName ON $tableName ($columns)";
            $conn->exec($sql);
            return "تم إنشاء الفهرس $indexName على الجدول $tableName ($columns) بنجاح.";
        } else {
            return "الفهرس $indexName موجود بالفعل على الجدول $tableName.";
        }
    } catch (PDOException $e) {
        return "خطأ في إنشاء الفهرس $indexName: " . $e->getMessage();
    }
}

// دالة لإضافة فهارس جدول الطلبات
function addOrdersIndexes($conn) {
    $results = [];
    
    // فهرس للبحث حسب حالة الطلب
    $results[] = addIndexIfNotExists($conn, 'orders', 'idx_orders_status', 'status');
    
    // فهرس للبحث حسب تاريخ الإنشاء
    $results[] = addIndexIfNotExists($conn, 'orders', 'idx_orders_created_at', 'created_at');
    
    // فهرس للبحث حسب تاريخ تحميل الطلب
    $results[] = addIndexIfNotExists($conn, 'orders', 'idx_orders_upload_date', 'order_upload_date');
    
    // فهرس للبحث حسب معرف الفرع
    $results[] = addIndexIfNotExists($conn, 'orders', 'idx_orders_branch_id', 'branch_id');
    
    // فهرس للبحث حسب معرف المستخدم
    $results[] = addIndexIfNotExists($conn, 'orders', 'idx_orders_user_id', 'user_id');
    
    // فهرس مركب للبحث حسب تاريخ التحميل والحالة
    $results[] = addIndexIfNotExists($conn, 'orders', 'idx_orders_upload_date_status', 'order_upload_date, status');
    
    // فهرس مركب للبحث حسب تاريخ التحميل ومعرف المستخدم
    $results[] = addIndexIfNotExists($conn, 'orders', 'idx_orders_upload_date_user', 'order_upload_date, user_id');
    
    return $results;
}

// دالة لإضافة فهارس جدول الفروع
function addBranchesIndexes($conn) {
    $results = [];
    
    // فهرس للبحث حسب اسم الفرع
    $results[] = addIndexIfNotExists($conn, 'branches', 'idx_branches_name', 'name');
    
    return $results;
}

// دالة لإضافة فهارس جدول المستخدمين
function addUsersIndexes($conn) {
    $results = [];
    
    // فهرس للبحث حسب دور المستخدم
    $results[] = addIndexIfNotExists($conn, 'users', 'idx_users_role', 'role');
    
    return $results;
}

// إضافة الفهارس
$results = [];
$results['orders'] = addOrdersIndexes($conn);
$results['branches'] = addBranchesIndexes($conn);
$results['users'] = addUsersIndexes($conn);

// عرض النتائج
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إضافة فهارس قاعدة البيانات</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            margin: 20px;
            background-color: #f5f5f5;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background-color: #fff;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
        }
        h1 {
            color: #333;
            text-align: center;
            margin-bottom: 20px;
        }
        .success {
            color: #155724;
            background-color: #d4edda;
            border: 1px solid #c3e6cb;
            padding: 10px;
            margin-bottom: 10px;
            border-radius: 4px;
        }
        .error {
            color: #721c24;
            background-color: #f8d7da;
            border: 1px solid #f5c6cb;
            padding: 10px;
            margin-bottom: 10px;
            border-radius: 4px;
        }
        .info {
            color: #0c5460;
            background-color: #d1ecf1;
            border: 1px solid #bee5eb;
            padding: 10px;
            margin-bottom: 10px;
            border-radius: 4px;
        }
        .section {
            margin-bottom: 20px;
            padding: 15px;
            background-color: #f9f9f9;
            border-radius: 4px;
        }
        .section h2 {
            margin-top: 0;
            color: #333;
            border-bottom: 1px solid #ddd;
            padding-bottom: 10px;
        }
        .back-link {
            display: block;
            text-align: center;
            margin-top: 20px;
            color: #007bff;
            text-decoration: none;
        }
        .back-link:hover {
            text-decoration: underline;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>إضافة فهارس قاعدة البيانات</h1>
        
        <div class="section">
            <h2>فهارس جدول الطلبات</h2>
            <?php foreach ($results['orders'] as $result): ?>
                <div class="<?php echo strpos($result, 'خطأ') !== false ? 'error' : 'success'; ?>">
                    <?php echo $result; ?>
                </div>
            <?php endforeach; ?>
        </div>
        
        <div class="section">
            <h2>فهارس جدول الفروع</h2>
            <?php foreach ($results['branches'] as $result): ?>
                <div class="<?php echo strpos($result, 'خطأ') !== false ? 'error' : 'success'; ?>">
                    <?php echo $result; ?>
                </div>
            <?php endforeach; ?>
        </div>
        
        <div class="section">
            <h2>فهارس جدول المستخدمين</h2>
            <?php foreach ($results['users'] as $result): ?>
                <div class="<?php echo strpos($result, 'خطأ') !== false ? 'error' : 'success'; ?>">
                    <?php echo $result; ?>
                </div>
            <?php endforeach; ?>
        </div>
        
        <div class="info">
            تم الانتهاء من إضافة الفهارس اللازمة لتحسين أداء قاعدة البيانات.
        </div>
        
        <a href="index.php" class="back-link">العودة إلى الصفحة الرئيسية</a>
    </div>
</body>
</html>
